﻿Imports System.Net

Namespace ESRI.BAOnline.SOAP
    ''' <summary>
    ''' The base abstract implementation of the <see cref="ITokenProvider"/> interface.
    ''' </summary>
    Public MustInherit Class TokenProvider
        Implements ITokenProvider
        Private token As String = Nothing
        Shared m_useTrustedSSLConnection As Boolean = True

#Region "ITokenProvider Members"

        ''' <summary>
        ''' Gets or sets the current authentication token.
        ''' </summary>
        ''' <remarks>If the current token is null or empty, the first get operation applies
        ''' the UpdateToken method.</remarks>
        Public Property CurrentToken() As String Implements ITokenProvider.CurrentToken
            Get
                If String.IsNullOrEmpty(token) Then
                    UpdateToken()
                End If
                Return token
            End Get
            Set(ByVal value As String)
                token = value
            End Set
        End Property


        ''' <summary>
        ''' Creates a new authentication token.
        ''' </summary>
        Public MustOverride Sub UpdateToken() Implements ITokenProvider.UpdateToken

#End Region

#Region "Common Members"

        ''' <summary>
        ''' Sets whether to use a trusted SSL connection in the GetToken method
        ''' or to allow not trusted connection also.
        ''' </summary>
        ''' <remarks>In the developer's environment, the SSL sertificate is often "not trusted".
        ''' False value of this property turns off the validation of the SSL sertificate when
        ''' requests a new token. Default value is true.</remarks>
        Public Shared WriteOnly Property UseTrustedSSLConnection() As Boolean
            Set(ByVal value As Boolean)
                m_useTrustedSSLConnection = value
            End Set
        End Property

        ''' <summary>
        ''' Gets a new token.
        ''' </summary>
        ''' <remarks>
        ''' <para>The token service URL must use the Secure Hypertext Transfer Protocol (HTTPS)
        ''' Secure Sockets Layer (SSL) connection. The query string contains the request type and
        ''' user's authentication credentials, for example,
        ''' "request=getToken&amp;username=...&amp;password=...".</para>
        ''' <para>If the <c>UseTrustedSSLConnection</c> property is false, a non-trusted SSL sertificate
        ''' is allowed. 
        ''' </para>
        ''' </remarks>
        ''' <param name="tokenServiceUrl">URL of the token service.</param>
        ''' <param name="query">Query string.</param>
        ''' <returns>The new token.</returns>
        Public Shared Function GetToken(ByVal tokenServiceUrl As String, ByVal query As String) As String
            Dim uriBuilder As New UriBuilder(tokenServiceUrl)
            uriBuilder.Query = query
            Dim request As System.Net.WebRequest = System.Net.WebRequest.Create(uriBuilder.Uri)
            Dim response As System.Net.WebResponse

            If m_useTrustedSSLConnection Then
                response = request.GetResponse()
            Else
                ' Change the sertificate validation callback
                Dim callback As System.Net.Security.RemoteCertificateValidationCallback = ServicePointManager.ServerCertificateValidationCallback
                ServicePointManager.ServerCertificateValidationCallback = AddressOf NoCertificateValidationCallback

                response = request.GetResponse()

                ' Restore the sertificate validation callback
                ServicePointManager.ServerCertificateValidationCallback = callback
            End If

            Return New System.IO.StreamReader(response.GetResponseStream()).ReadToEnd()
        End Function

        ''' <summary>
        ''' This callback method trusts to any SSL sertificate. 
        ''' </summary>
        ''' <param name="sender">An object that contains state information for this validation.</param>
        ''' <param name="certificate">The certificate used to authenticate the remote party.</param>
        ''' <param name="chain">The chain of certificate authorities associated with the remote certificate.</param>
        ''' <param name="errors">One or more errors associated with the remote certificate.</param>
        ''' <returns>True value.</returns>
        Public Shared Function NoCertificateValidationCallback(ByVal sender As Object, ByVal certificate As System.Security.Cryptography.X509Certificates.X509Certificate, ByVal chain As System.Security.Cryptography.X509Certificates.X509Chain, ByVal errors As System.Net.Security.SslPolicyErrors) As Boolean
            Return certificate IsNot Nothing
        End Function

#End Region
    End Class
End Namespace