﻿Imports System.Net
Imports Samples_VB.BAOReport_Proxy


Namespace ESRI.BAOnline.SOAP
    ''' <summary>
    ''' A helper class to the BAOReport client.
    ''' </summary>
    ''' <remarks>
    ''' Every method of the BAOReport client is executed within the try-catch block and,
    ''' if the token is expired, a new token is automatically requested and the
    ''' method is executed once more.
    ''' </remarks>
    Public Class BAOReportHelper
        Private baoReport As New BAOReport()
        Private tokenProvider As ITokenProvider

#Region "Constructor"

        ''' <summary>
        ''' Constructs a new instance of the helper class.
        ''' </summary>
        ''' <param name="tokenProvider">A provider of an authentication token.</param>
        Public Sub New(ByVal tokenProvider As ITokenProvider)
            Me.tokenProvider = tokenProvider
        End Sub

#End Region

#Region "Common Members"

        ''' <summary>
        ''' Gets an instance of the BAOReport client associated with this helper.
        ''' </summary>
        Public ReadOnly Property Client() As BAOReport
            Get
                Return baoReport
            End Get
        End Property

        ''' <summary>
        ''' Gets the current value of the authentication token.
        ''' </summary>
        Public ReadOnly Property Token() As String
            Get
                Return tokenProvider.CurrentToken
            End Get
        End Property

        ''' <summary>
        ''' Sets values for analysis parameters.
        ''' </summary>
        ''' <param name="parameters">Analysis parameters.</param>
        Public Sub SetAnalysisParameters(ByVal parameters As AnalysisParameters)
            parameters.token = Token
        End Sub

        ''' <summary>
        ''' Tests the exception status code to be the "Token Expired" event.
        ''' </summary>
        ''' <param name="ex">An exception.</param>
        ''' <returns>True, if the "Token Expired" event is recognized.</returns>
        Public Function IsTokenExpired(ByVal ex As Exception) As Boolean
            Return ex.Message.Contains("498")
        End Function

        ''' <summary>
        ''' Updates the authentication token.
        ''' </summary>
        Public Sub UpdateToken()
            tokenProvider.UpdateToken()
        End Sub

#End Region

#Region "Static Helper Members"

        ''' <summary>
        ''' Internal constant used in tasks producing a report only.
        ''' </summary>
        Shared ReportOutput As TaskOutputType() = {TaskOutputType.GetReport}

#End Region

#Region "Proxy method for GeofenceInfo Task"

        ''' <summary>
        ''' Requests geography features available for the current user account.
        ''' </summary>
        ''' <remarks>
        ''' <para>
        ''' This task queries available features from a standard geography data layer according
        ''' to the Business Analyst Online API user account geofence constraints.
        ''' </para>
        ''' <para>
        ''' The standard geography data layer is specified by the <c>geoLevelID</c> parameter value.
        ''' The query criteria is specified by the <c>searchString</c> parameter value.
        ''' This query is performed on the name and ID attributes of every available feature
        ''' in the standard geography layer. All available features in the specified standard
        ''' geography layer are returned when no value for the <c>searchString</c> parameter is specified
        ''' in the request.
        ''' </para>
        ''' <para>
        ''' When specifying the request with a Business Analyst Online API user account with geofence
        ''' constraints, the geofence criteria is applied before the output response is generated.
        ''' The output response will contain a list of feature IDs and names in a 
        ''' GeoFenceInfoOutput.StdGeographiesFeatures array. If no matching features were found by
        ''' the query criteria or all of the features were rejected by the geofence constraints,
        ''' the array will be null.
        ''' </para>
        ''' <para>
        ''' Additionally, a map image and a feature class can be generated for resulting features and
        ''' returned in the output results depending on the output options selected. If you only need
        ''' feature IDs and names as the output, you can pass null as the <c>outputOptions</c> parameter.
        ''' </para>
        ''' <para>
        ''' NOTE: A user account is identified by the authentication token parameter value.
        ''' </para>
        ''' </remarks>
        ''' <param name="geoLevelId">Geography level ID.</param>
        ''' <param name="searchString">A string restricting the search in geography feature IDs and names.</param>
        ''' <param name="outputOptions">Array of task output options.
        ''' Options for this method include creating the feature class for subsequent analysis (GetFeatureClass).</param>
        ''' <returns>GeoFence output results.</returns>
        Public Function GeofenceInfo(ByVal activeDatasetId As String, ByVal geoLevelId As String, ByVal searchString As String, ByVal outputOptions As TaskOutputType()) As GeofenceInfoOutput
            Try
                Return Client.GeofenceInfo(activeDatasetId, geoLevelId, searchString, outputOptions, Token)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                Return Client.GeofenceInfo(activeDatasetId, geoLevelId, searchString, outputOptions, Token)
            End Try
        End Function

#End Region

#Region "Proxy methods for Report Tasks"

        ''' <summary>
        ''' A comparative report that benchmarks two or more trade areas based on selected
        ''' volumetric data (typically demographic data).
        ''' </summary>
        ''' <remarks>
        ''' <note type="caution">
        ''' The signature of this proxy-method is simplified by removing the <c>OutputType[]</c> parameter
        ''' which always contains the <c>OutputType.GetReport</c> option only.</note>
        ''' </remarks>
        ''' <param name="parameters">Configuration options for analysis.</param>
        ''' <returns>Task output results.</returns>
        Public Function BenchmarkReport(ByVal parameters As BenchmarkReportParameters) As TaskResultOutput
            Try
                SetAnalysisParameters(parameters)
                Return Client.BenchmarkReport(parameters, ReportOutput)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                SetAnalysisParameters(parameters)
                Return Client.BenchmarkReport(parameters, ReportOutput)
            End Try
        End Function

        ''' <overloads>Generates summary reports for trade areas.</overloads>
        ''' <summary>
        ''' Generates summary reports for trade areas.
        ''' </summary>
        ''' <remarks>
        ''' <note type="caution">
        ''' This proxy-method generates reports only.
        ''' </note>
        ''' </remarks>
        ''' <param name="parameters">Configuration options for analysis.</param>
        ''' <returns>Task output results.</returns>
        Public Function SummaryReports(ByVal parameters As SummaryReportsParameters) As TaskResultOutput
            Return SummaryReports(parameters, ReportOutput)
        End Function

        ''' <summary>
        ''' Generates summary reports for trade areas.
        ''' </summary>
        ''' <param name="parameters">Configuration options for analysis.</param>
        ''' <param name="outputOptions">Array of task output options.
        ''' Options for this method include creating a report (GetReport).</param>
        ''' <returns>Task output results.</returns>
        Public Function SummaryReports(ByVal parameters As SummaryReportsParameters, ByVal outputOptions As TaskOutputType()) As TaskResultOutput
            Try
                SetAnalysisParameters(parameters)
                Return Client.SummaryReports(parameters, outputOptions)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                SetAnalysisParameters(parameters)
                Return Client.SummaryReports(parameters, outputOptions)
            End Try
        End Function

#End Region

#Region "Proxy methods for Trade Area Tasks"

        ''' <summary>
        ''' Creates trade areas based on the driving time or driving distance around store features.
        ''' </summary>
        ''' <param name="parameters">Configuration options for analysis.</param>
        ''' <param name="imageOptions">Configuration options for rendering output when
        ''' GetMapImage option is specified in the outputOptions parameter.</param>
        ''' <param name="outputOptions">Array of task output options.
        ''' Options for this method include creating the feature class for subsequent analysis (GetFeatureClass),
        ''' and creating a report (GetReport).</param>
        ''' <returns>Task output results.</returns>
        Public Function DriveTime(ByVal parameters As DriveTimeParameters, ByVal imageOptions As RenderingParameters, ByVal outputOptions As TaskOutputType()) As TaskResultOutput
            Try
                SetAnalysisParameters(parameters)
                Return Client.DriveTime(parameters, imageOptions, outputOptions)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                SetAnalysisParameters(parameters)
                Return Client.DriveTime(parameters, imageOptions, outputOptions)
            End Try
        End Function

        ''' <summary>
        ''' Creates a new feature class of ring-based trade area using a set of radii.
        ''' </summary>
        ''' <param name="parameters">Configuration options for analysis.</param>
        ''' <param name="imageOptions">Configuration options for rendering output when
        ''' GetMapImage option is specified in the outputOptions parameter.</param>
        ''' <param name="outputOptions">Array of task output options.
        ''' Options for this method include creating the feature class for subsequent analysis (GetFeatureClass),
        ''' and creating a report (GetReport).</param>
        ''' <returns>Task output results.</returns>
        Public Function SimpleRings(ByVal parameters As SimpleRingsParameters, ByVal imageOptions As RenderingParameters, ByVal outputOptions As TaskOutputType()) As TaskResultOutput
            Try
                SetAnalysisParameters(parameters)
                Return Client.SimpleRings(parameters, imageOptions, outputOptions)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                SetAnalysisParameters(parameters)
                Return Client.SimpleRings(parameters, imageOptions, outputOptions)
            End Try
        End Function

#End Region

#Region "Proxy methods for Lookup Utilities"

        ''' <summary>
        ''' Gets the list of all available report templates for summary reports.
        ''' </summary>
        ''' <returns>Array of report templates.</returns>
        Public Function GetReportTemplates(ByVal activeDatasetId As String) As ReportTemplateInfo()
            Try
                Return Client.GetReportTemplates(activeDatasetId, Token)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                Return Client.GetReportTemplates(activeDatasetId, Token)
            End Try
        End Function

        ''' <summary>
        ''' Gets IDs and names of standard geography levels.
        ''' </summary>
        ''' <returns>Array of IDs and names of the standard geography levels.</returns>
        Public Function GetStandardGeographyLevels(ByVal activeDatasetId As String) As IDNamePair()
            Try
                Return Client.GetStandardGeographyLevels(activeDatasetId, Token)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                Return Client.GetStandardGeographyLevels(activeDatasetId, Token)
            End Try
        End Function

        ''' <summary>
        ''' Gets a list of Business Analyst summarization variables and their metadata which are accessible
        ''' with the Business Analyst Online API subscriber account.
        ''' </summary>
        ''' <returns>Array containing all summarization fields available for the given account.</returns>
        Public Function GetSummarizations(ByVal activeDatasetId As String) As SummarizationInfo()
            Try
                Return Client.GetSummarizations(activeDatasetId, Token)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                Return Client.GetSummarizations(activeDatasetId, Token)
            End Try
        End Function

#End Region

#Region "Proxy methods for Standard Geography Utilities"

        ''' <summary>
        ''' Returns IDs and names of features from specified geography levels that match the given location.
        ''' </summary>
        ''' <param name="parameters">Geographic search parameters.</param>
        ''' <returns>IDs and names of features of geography levels matching the given location.</returns>
        Public Function ReturnStdGeography(ByVal parameters As ReturnStdGeographyParams) As StdGeographiesReturnedFeature()
            SetAnalysisParameters(parameters)

            Try
                Return Client.ReturnStdGeography(parameters)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                Return Client.ReturnStdGeography(parameters)
            End Try
        End Function

        ''' <summary>
        ''' Geography location names and IDs are returned for a string-based query.
        ''' </summary>
        ''' <param name="parameters">Geographic search parameters.</param>
        ''' <returns>Array of features that match the query.</returns>
        Public Function StdGeographiesByAttributes(ByVal parameters As StdGeographiesByAttributesParams) As StdGeographiesReturnedFeature()
            Try
                SetAnalysisParameters(parameters)
                Return Client.StdGeographiesByAttributes(parameters)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                SetAnalysisParameters(parameters)
                Return Client.StdGeographiesByAttributes(parameters)
            End Try
        End Function

        ''' <summary>
        ''' Geography location names and IDs are returned for a given extent.
        ''' </summary>
        ''' <param name="parameters">Geographic search parameters.</param>
        ''' <returns>Array of features that match the query.</returns>
        Public Function StdGeographiesFromExtent(ByVal parameters As StdGeographiesFromExtentParams) As StdGeographiesReturnedFeature()
            Try
                SetAnalysisParameters(parameters)
                Return Client.StdGeographiesFromExtent(parameters)
            Catch ex As Exception
                If Not IsTokenExpired(ex) Then
                    Throw
                End If
                UpdateToken()
                SetAnalysisParameters(parameters)
                Return Client.StdGeographiesFromExtent(parameters)
            End Try
        End Function

#End Region
    End Class
End Namespace